using System;
using System.Collections;
using System.Data;
using System.Data.SqlClient;
using System.Reflection;
using gov.va.med.vbecs.ExceptionManagement;
using STOREDPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;
using PATIENTTRANSFUSION = gov.va.med.vbecs.Common.VbecsTables.PatientTransfusion;
using gov.va.med.vbecs.Common;

namespace gov.va.med.vbecs.DAL
{
		#region Header

		///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
		///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
		///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
		///<summary>
		///	Data access component implementing data persistence logic for
		///	patient transfusion related data.
		///</summary>

		#endregion

	public class PatientTransfusion
	{
		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/8/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4081"> 
		///		<ExpectedInput>DataTables</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4082"> 
		///		<ExpectedInput>Invalid RowVersion</ExpectedInput>
		///		<ExpectedOutput>UniqueDatabaseConstraintViolatedException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Saves the PatientTransfusion record
		/// </summary>
		/// <param name="dtPatientTransfusion"></param>
		/// <param name="dtComment"></param>
		/// <param name="dtWorkload"></param>
		/// <param name="dtExceptionsAboIncompatible"></param>
		/// <returns></returns>
		public static bool Save(DataTable dtPatientTransfusion, DataTable dtWorkload, DataTable dtComment, DataTable dtExceptionsAboIncompatible)
		{
			ArrayList spArray = new ArrayList();
			ArrayList dtArray = new ArrayList();

			// Not needed since this is new!
			if (dtPatientTransfusion.Columns.Contains(PATIENTTRANSFUSION.RowVersion))
			{
				dtPatientTransfusion.Columns.Remove(PATIENTTRANSFUSION.RowVersion);
			}

			spArray.Add(Common.VbecsStoredProcs.InsertPatientTransfusion.StoredProcName);	
			dtArray.Add(dtPatientTransfusion);

			if ((dtExceptionsAboIncompatible != null) && (dtExceptionsAboIncompatible.Rows.Count>0))
			{
				spArray.Add(Common.VbecsStoredProcs.InsertExceptionAboIncompatibleTransfusion.StoredProcName);
				dtArray.Add(dtExceptionsAboIncompatible);
			}

			if ((dtWorkload != null) && (dtWorkload.Rows.Count>0))
			{
				spArray.Add(Common.VbecsStoredProcs.InsertWorkloadEvents.StoredProcName);
				dtArray.Add(dtWorkload);
			}

			// Need to save comment AFTER we insert the PT!
			if ((dtComment != null) && (dtComment.Rows.Count>0))
			{
				spArray.Add(Common.VbecsStoredProcs.InsertPatientTransfusionComment.StoredProcName);
				dtArray.Add(dtComment);
			}

			return (new Common.StoredProcedure().TransactionalGetValue(spArray,dtArray)==0);
		}


		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/8/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4083"> 
		///		<ExpectedInput>Updates a valid Patient Transfusion</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4084"> 
		///		<ExpectedInput>Invalid Data</ExpectedInput>
		///		<ExpectedOutput>RowVersionException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Updates the patient transfusion record
		/// </summary>
		/// <param name="dtPatientTransfusion"></param>
		/// <param name="dtComment"></param>
		/// <param name="newComment"></param>
		/// <param name="dtExceptionsAboIncompatible"></param>
		/// <param name="dtExceptionsElectronicTransfusion"></param>
		/// <returns></returns>
		public static bool Update(DataTable dtPatientTransfusion, DataTable dtComment, bool newComment, DataTable dtExceptionsAboIncompatible, DataTable dtExceptionsElectronicTransfusion)
		{
			ArrayList spArray = new ArrayList();
			ArrayList dtArray = new ArrayList();

			spArray.Add(Common.VbecsStoredProcs.UpdatePatientTransfusionRecord.StoredProcName);
			dtArray.Add(dtPatientTransfusion);

			if (dtComment != null)
			{
				if (newComment)	spArray.Add(Common.VbecsStoredProcs.InsertPatientTransfusionComment.StoredProcName);
				else spArray.Add(Common.VbecsStoredProcs.UpdatePatientTransfusionComment.StoredProcName);

				dtArray.Add(dtComment);
			}

			if(dtExceptionsAboIncompatible != null)
			{
				spArray.Add(Common.VbecsStoredProcs.InsertExceptionAboIncompatibleTransfusion.StoredProcName);
				dtArray.Add(dtExceptionsAboIncompatible);
			}

			// CR 2942
			if(dtExceptionsElectronicTransfusion != null)
			{
				spArray.Add(Common.VbecsStoredProcs.InsertExceptionElectronicTransfusion.StoredProcName);
				dtArray.Add(dtExceptionsElectronicTransfusion);
			}

			return ((new StoredProcedure()).TransactionalGetValue(spArray,dtArray)==0);
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/8/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2147"> 
		///		<ExpectedInput>PatientTransfuion Guid</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2148"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Returns comment associated with a patient transfusion
		/// </summary>
		/// <param name="patientTransfusionGuid"></param>
		/// <returns></returns>
		public static DataTable GetTransfusionComment(Guid patientTransfusionGuid)
		{
			if (patientTransfusionGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("Patient unique identifier").ResString);
			}

			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetPatientTransfusionComment.patienttransfusionguid, SqlDbType.UniqueIdentifier)
			};

			prms[0].Value = patientTransfusionGuid;

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetPatientTransfusionComment.StoredProcName, prms);

			return ds.Tables[0];
		}


		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/8/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4085"> 
		///		<ExpectedInput>A patient (GUID) that has issued units</ExpectedInput>
		///		<ExpectedOutput>DataTable with > 0 rows</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4086"> 
		///		<ExpectedInput>Empty Guid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetIssuedAndTransfusedUnitsForPatient
		/// </summary>
		/// <param name="patientGuid">patientGuid</param>
		/// <returns></returns>
		public static DataTable GetIssuedAndTransfusedUnitsForPatient(System.Guid patientGuid)
		{
			if (patientGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("Patient unique identifier").ResString);
			}

			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetIssuedAndTransfusedUnitsForPatient.patientguid, SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetIssuedAndTransfusedUnitsForPatient.divisioncode, SqlDbType.Char)
			};

			prms[0].Value = patientGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetIssuedAndTransfusedUnitsForPatient.StoredProcName, prms);

			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/8/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4087"> 
		///		<ExpectedInput>A patient (GUID) and unit (GUID) that are associated with a 
		///		Patient Transfusion record.</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4088"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetPatientTransfusionRecord
		/// </summary>
		/// <param name="patientGuid"></param>
		/// <param name="bloodUnitGuid"></param>
		/// <returns></returns>
		public static DataTable GetPatientTransfusionRecord(Guid patientGuid, Guid bloodUnitGuid)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetPatientTransfusionRecord.patientguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetPatientTransfusionRecord.bloodunitguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetPatientTransfusionRecord.divisioncode, System.Data.SqlDbType.Char)
			};
			
			prms[0].Value = patientGuid;
			prms[1].Value = bloodUnitGuid;
			prms[2].Value = Common.LogonUser.LogonUserDivisionCode;

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetPatientTransfusionRecord.StoredProcName, prms);
			
			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/8/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2149"> 
		///		<ExpectedInput>BloodUnit Guid</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2150"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetPatientTransfusionRecord
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <returns></returns>
		public static DataTable GetPatientTransfusionRecord(Guid bloodUnitGuid)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetPatientTransfusionRecord.bloodunitguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetPatientTransfusionRecord.divisioncode, System.Data.SqlDbType.Char)
			};
			
			prms[0].Value = bloodUnitGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetPatientTransfusionRecord.StoredProcName, prms);
			
			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4161"> 
		///		<ExpectedInput>Valid PatientGuid</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4162"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetRecentlyIssuedAndTransfusedUnitsForPatient
		/// </summary>
		/// <param name="patientGuid"></param>
		/// <returns></returns>
		public static DataTable GetRecentlyIssuedAndTransfusedUnitsForPatient(System.Guid patientGuid)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetRecentlyIssuedAndTransfusedUnitsForPatient.divisioncode, System.Data.SqlDbType.Char),
				new SqlParameter(STOREDPROC.GetRecentlyIssuedAndTransfusedUnitsForPatient.patientguid, System.Data.SqlDbType.UniqueIdentifier),
			};
			
			prms[0].Value = Common.LogonUser.LogonUserDivisionCode;
			prms[1].Value = patientGuid;
			
           
			System.Data.DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetRecentlyIssuedAndTransfusedUnitsForPatient.StoredProcName, prms);

			return(ds.Tables[0]);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/9/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8183"> 
		///		<ExpectedInput>Valid patient guid</ExpectedInput>
		///		<ExpectedOutput>Count of issued and transfused units</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8184"> 
		///		<ExpectedInput>Invalid patient guid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException thrown</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets a count of recently issued and/or transfused units for a patient
		/// </summary>
		/// <param name="patientGuid">Patient Guid</param>
		/// <returns>Count of issued and/or transfused units</returns>
		public static int GetRecentlyIssuedAndTransfusedUnitsForPatientCount(System.Guid patientGuid)
		{
			if (patientGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("Patient unique identifier").ResString);
			}

			int count = 0;

			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetRecentlyIssuedAndTransfusedUnitsForPatientCount.patientguid, SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetRecentlyIssuedAndTransfusedUnitsForPatientCount.divisioncode, SqlDbType.Char)
			};

			prms[0].Value = patientGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetRecentlyIssuedAndTransfusedUnitsForPatientCount.StoredProcName, prms);
			foreach(DataTable dt in ds.Tables)
			{
				foreach(DataRow dr in dt.Rows)
				{
					count += (int)dr["Total"];
				}
			}
			//
			return count;
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2783"> 
		///		<ExpectedInput>Valid patient GUID</ExpectedInput>
		///		<ExpectedOutput>boolean</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2784"> 
		///		<ExpectedInput>Invalid patient GUID</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>
		///	Checks transfusion history within last 3 months 
		///	and verifies that it exists
		///</summary>
		///<param name="patientGuid">Patient GUID</param>
		///<returns>True if transfusion history exists. Otherwise - false.</returns>
		public static bool IsTransfusionHistoryforPatient(Guid patientGuid)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.IsTransfusionHistoryforPatient.patientguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.IsTransfusionHistoryforPatient.divisioncode, System.Data.SqlDbType.VarChar)
			};
			
			prms[0].Value = patientGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;
           
			return new StoredProcedure().GetBoolReturnValue( STOREDPROC.IsTransfusionHistoryforPatient.StoredProcName, prms );	
		}

		
		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/8/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4089"> 
		///		<ExpectedInput>A patient (GUID) and unit (GUID) that are associated with a 
		///		Patient Transfusion record.</ExpectedInput>
		///		<ExpectedOutput>DataRow</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4090"> 
		///		<ExpectedInput>Invalid parameters</ExpectedInput>
		///		<ExpectedOutput>null</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetTransfusionRelatedInformation
		/// </summary>
		/// <param name="patientGuid"></param>
		/// <param name="bloodUnitGuid"></param>
		/// <returns></returns>
		public static DataRow GetTransfusionRelatedInformation(Guid patientGuid, Guid bloodUnitGuid)
		{
			SqlParameter[] prms =
					{
						new SqlParameter(STOREDPROC.GetTransfusionRelatedInformation.patientguid, System.Data.SqlDbType.UniqueIdentifier),
						new SqlParameter(STOREDPROC.GetTransfusionRelatedInformation.bloodunitguid, System.Data.SqlDbType.UniqueIdentifier),
						new SqlParameter(STOREDPROC.GetTransfusionRelatedInformation.divisioncode, System.Data.SqlDbType.VarChar)
					};
		
			prms[0].Value = patientGuid;
			prms[1].Value = bloodUnitGuid;
			prms[2].Value = Common.LogonUser.LogonUserDivisionCode;

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetTransfusionRelatedInformation.StoredProcName, prms);

			if (ds.Tables[0].Rows.Count > 0) return ds.Tables[0].Rows[0];
			else return null;
		}
		
		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/8/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2151"> 
		///		<ExpectedInput>BloodUnit Guid</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2152"> 
		///		<ExpectedInput>Invalid guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Return the transfused patient data
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <returns></returns>
		public static DataTable GetTransfusedPatient(Guid bloodUnitGuid)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetTransfusedPatient.BloodUnitGuid, System.Data.SqlDbType.UniqueIdentifier)
			};
			
			prms[0].Value = bloodUnitGuid;
           
			DataSet ds = StoredProcedure.GetData(Common.VbecsStoredProcs.GetTransfusedPatient.StoredProcName, prms);
			if (ds.Tables.Count > 0)
				return ds.Tables[0];
			else
				return new DataTable();
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/8/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2155"> 
		///		<ExpectedInput>Patient Guid, Date Range</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2156"> 
		///		<ExpectedInput>Invalid parameters</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Return the history of transfusions for patient
		/// </summary>
		/// <param name="patientGuid"></param>
		/// <param name="startDate"></param>
		/// <param name="endDate"></param>
		/// <returns></returns>
		public static DataTable GetTransfusionHistoryForPatient(Guid patientGuid, DateTime startDate, DateTime endDate)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetTransfusionHistoryForPatient.patientguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetTransfusionHistoryForPatient.startdate, System.Data.SqlDbType.DateTime),
				new SqlParameter(STOREDPROC.GetTransfusionHistoryForPatient.enddate, System.Data.SqlDbType.DateTime),
				new SqlParameter(STOREDPROC.GetTransfusionHistoryForPatient.divisioncode, System.Data.SqlDbType.VarChar)
			};
			
			prms[0].Value = patientGuid;
			prms[1].Value = startDate;
			prms[2].Value = endDate;
			prms[3].Value = Common.LogonUser.LogonUserDivisionCode;
           
			DataSet ds = StoredProcedure.GetData(Common.VbecsStoredProcs.GetTransfusionHistoryForPatient.StoredProcName, prms);
			if (ds.Tables.Count > 0)
				return ds.Tables[0];
			else
				return new DataTable();
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/24/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4461"> 
		///		<ExpectedInput>Valid patient specimen unique identifier</ExpectedInput>
		///		<ExpectedOutput>Data table of transfused units</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4462"> 
		///		<ExpectedInput>Invalid patient specimen unique identifier</ExpectedInput>
		///		<ExpectedOutput>Argument exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Static function returns a data table of units transfused for a given specimen
		/// </summary>
		/// <param name="patientSpecimenGuid">Patient specimen unique identifier</param>
		/// <returns>Data table of transfused units</returns>
		public static DataTable GetTransfusionHistoryForPatientSpecimen(Guid patientSpecimenGuid)
		{
			if (patientSpecimenGuid==Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("Patient specimen unique identifier").ResString);
			}
			//
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetTransfusionHistoryForPatientSpecimen.patientspecimenguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetTransfusionHistoryForPatientSpecimen.divisioncode, System.Data.SqlDbType.VarChar)
			};
			
			prms[0].Value = patientSpecimenGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;
           
			DataSet ds = StoredProcedure.GetData(Common.VbecsStoredProcs.GetTransfusionHistoryForPatientSpecimen.StoredProcName, prms);

			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>D. Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/17/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4656"> 
		///		<ExpectedInput>Valid PatientGuid</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4657"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets transfused blood units (guid and product) 
		/// for a Patient
		/// </summary>
		/// <param name="patientGuid"></param>
		/// <param name="divisionCode"></param>
		/// <returns>DataTable</returns>
		public static DataTable GetTransfusedUnitsForPatient(Guid patientGuid, string divisionCode)
		{
			if (patientGuid.Equals(Guid.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("patient unique identifier").ResString);
			}
			
			if (divisionCode == null || divisionCode.Equals(string.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("division code").ResString);
			}
			
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetTransfusedUnitsForPatient.patientguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetTransfusedUnitsForPatient.divisioncode, System.Data.SqlDbType.Char)
			};
			
			prms[0].Value = patientGuid;
			prms[1].Value = divisionCode;
			
			return Common.StoredProcedure.GetData(STOREDPROC.GetTransfusedUnitsForPatient.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>D. Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/17/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4757"> 
		///		<ExpectedInput>Valid PatientGuid</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4758"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets transfused blood units (guid and product) 
		/// for a Patient
		/// </summary>
		/// <param name="patientGuid"></param>
		/// <param name="divisionCode"></param>
		/// <returns>DataTable</returns>
		public static DataTable GetPatientTransfusionRecords(Guid patientGuid, string divisionCode)
		{
			if (patientGuid.Equals(Guid.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("patient unique identifier").ResString);
			}
			
			if (divisionCode == null || divisionCode.Equals(string.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("division code").ResString);
			}
			
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetPatientTransfusionRecords.patientguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetPatientTransfusionRecords.divisioncode, System.Data.SqlDbType.Char)
			};
			
			prms[0].Value = patientGuid;
			prms[1].Value = divisionCode;
			
			return Common.StoredProcedure.GetData(STOREDPROC.GetPatientTransfusionRecords.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>D. Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/17/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="2157"> 
		///		<ExpectedInput>Valid DateRange/Division</ExpectedInput>
		///		<ExpectedOutput>int</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2158"> 
		///		<ExpectedInput>Invalid Division</ExpectedInput>
		///		<ExpectedOutput>0</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetTransfusedUnitsInDateRangeCount
		/// </summary>
		/// <param name="startDate"></param>
		/// <param name="endDate"></param>
		/// <param name="divisionCode"></param>
		/// <param name="patientGuid"></param>
		/// <returns></returns>
		public static int GetTransfusedUnitsInDateRangeCount(DateTime startDate, DateTime endDate, string divisionCode, Guid patientGuid)
		{
			int count = 0;
			//
			SqlParameter [] prms = 
				{
					new SqlParameter(STOREDPROC.GetTransfusedUnitsInDateRangeCount.startdate, System.Data.SqlDbType.DateTime),
					new SqlParameter(STOREDPROC.GetTransfusedUnitsInDateRangeCount.enddate, System.Data.SqlDbType.DateTime),
					new SqlParameter(STOREDPROC.GetTransfusedUnitsInDateRangeCount.patientguid, System.Data.SqlDbType.UniqueIdentifier),
					new SqlParameter(STOREDPROC.GetTransfusedUnitsInDateRangeCount.divisioncode, System.Data.SqlDbType.Char)
				};
			prms[0].Value = startDate;
			prms[1].Value = endDate;
			if (patientGuid != Guid.Empty)
			{
				prms[2].Value = patientGuid;
			}
			else
			{
				prms[2].Value = DBNull.Value;
			}
			prms[3].Value = divisionCode;
			//
			DataTable dt = StoredProcedure.GetData(STOREDPROC.GetTransfusedUnitsInDateRangeCount.StoredProcName, prms).Tables[0];
			foreach(DataRow dr in dt.Rows)
			{
				count += (int)dr["Total"];
			}
			//
			return count;
		}

		///<Developers>
		///	<Developer>D. Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/17/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="2161"> 
		///		<ExpectedInput>Valid DateRange/Division</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2162"> 
		///		<ExpectedInput>Invalid Division</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetTransfusedUnitsInDateRange
		/// </summary>
		/// <param name="startDate"></param>
		/// <param name="endDate"></param>
		/// <param name="divisionCode"></param>
		/// <param name="patientGuid"></param>
		/// <returns></returns>
		public static DataTable GetTransfusedUnitsInDateRange(DateTime startDate, DateTime endDate, string divisionCode, Guid patientGuid)
		{
			SqlParameter [] prms = 
				{
					new SqlParameter(STOREDPROC.GetTransfusedUnitsInDateRange.startdate, System.Data.SqlDbType.DateTime),
					new SqlParameter(STOREDPROC.GetTransfusedUnitsInDateRange.enddate, System.Data.SqlDbType.DateTime),
					new SqlParameter(STOREDPROC.GetTransfusedUnitsInDateRange.patientguid, System.Data.SqlDbType.UniqueIdentifier),
					new SqlParameter(STOREDPROC.GetTransfusedUnitsInDateRange.divisioncode, System.Data.SqlDbType.Char)
				};
			prms[0].Value = startDate;
			prms[1].Value = endDate;
			if (patientGuid != Guid.Empty)
			{
				prms[2].Value = patientGuid;
			}
			else
			{
				prms[2].Value = DBNull.Value;
			}
			prms[3].Value = divisionCode;
			//
			return StoredProcedure.GetData(STOREDPROC.GetTransfusedUnitsInDateRange.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>D. Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/17/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="2163"> 
		///		<ExpectedInput>Valid DateRange/Where Clause</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2164"> 
		///		<ExpectedInput>Invalid WhereClause</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Get the data for transfusion report
		/// </summary>
		/// <param name="startDate"></param>
		/// <param name="endDate"></param>
		/// <param name="whereClause"></param>
		/// <param name="divisionCode"></param>
		/// <param name="reportType"></param>
		/// <returns></returns>
		public static DataTable GetTransfusionReport(DateTime startDate, DateTime endDate, string whereClause,
			string divisionCode, int reportType)
		{
			SqlParameter [] prms = 
				{
					new SqlParameter(STOREDPROC.GetTransfusionReport.startdate, System.Data.SqlDbType.DateTime),
					new SqlParameter(STOREDPROC.GetTransfusionReport.enddate, System.Data.SqlDbType.DateTime),
					new SqlParameter(STOREDPROC.GetTransfusionReport.divisioncode, System.Data.SqlDbType.Char),
					new SqlParameter(STOREDPROC.GetTransfusionReport.searchstring, System.Data.SqlDbType.VarChar),
					new SqlParameter(STOREDPROC.GetTransfusionReport.reporttype, System.Data.SqlDbType.Int),
					new SqlParameter(STOREDPROC.GetTransfusionReport.delimiter, System.Data.SqlDbType.VarChar)
				};
			prms[0].Value = startDate;
			prms[1].Value = endDate;
			prms[2].Value = divisionCode;
			prms[3].Value = whereClause;
			prms[4].Value = reportType;
			prms[5].Value = "|";
			//
			return StoredProcedure.GetData(STOREDPROC.GetTransfusionReport.StoredProcName, prms).Tables[0];
		}
	}
}
